/*
 * Copyright (c) 2024, Azul Systems, Inc. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

import jdk.crac.Core;
import jdk.internal.crac.JDKFdResource;
import jdk.test.lib.Platform;
import jdk.test.lib.crac.CracBuilder;
import jdk.test.lib.crac.CracTest;
import jdk.test.lib.crac.CracTestArg;
import jdk.test.lib.process.*;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import static jdk.test.lib.Asserts.*;

/**
 * @test
 * @library /test/lib
 * @modules java.base/jdk.internal.crac:+open
 * @requires (os.family == "linux")
 * @build LoggingVMlogOpenTestNegative
 * @run driver jdk.test.lib.crac.CracTest true
 * @run driver jdk.test.lib.crac.CracTest false
 */
public class LoggingVMlogOpenTestNegative implements CracTest {

    @CracTestArg(0)
    boolean scenario1;

    @Override
    public void test() throws Exception {
        Path logPathO = Files.createTempFile(getClass().getName(), "-vmlog1.txt");
        Path logPathR = Files.createTempFile(getClass().getName(), "-vmlog2.txt");
        Files.deleteIfExists(logPathR);
        try {
            CracBuilder builder = new CracBuilder();
            builder.vmOption("-XX:+UnlockDiagnosticVMOptions");
            builder.vmOption("-XX:+LogVMOutput");
            builder.vmOption("-XX:LogFile=" + logPathO);
            builder.startCheckpoint().waitForCheckpointed();

            assertNotEquals(0L, Files.size(logPathO));
            Files.deleteIfExists(logPathO);

            builder.clearVmOptions();
            if (scenario1) {
                builder.vmOption("-XX:+UnlockDiagnosticVMOptions");
                builder.vmOption("-XX:LogFile=" + logPathR);
                builder.doRestore();
            } else {
                builder.vmOption("-XX:+UnlockDiagnosticVMOptions");
                builder.vmOption("-XX:+LogCompilation");
                builder.vmOption("-XX:LogFile=" + logPathR);
                OutputAnalyzer oa = builder.captureOutput(true).startRestore().outputAnalyzer().shouldContain("cannot be set during restore").shouldNotHaveExitValue(0);
            }
        } finally {
            assertFalse(Files.exists(logPathO));
            // isDebugBuild() due to: [!PRODUCT] use_vm_log() uses FLAG_IS_DEFAULT(LogFile)
            if (scenario1 && Platform.isDebugBuild()) {
                // As the CRaC restore is started later during JDK initialization the LogFile got already created.
                assertTrue(Files.exists(logPathR));
            } else {
                // Here a check of invalid parameters happens before the LogFile gets created.
                assertFalse(Files.exists(logPathR));
            }
        }
    }

    @Override
    public void exec() throws Exception {
        Core.checkpointRestore();
        System.out.println(RESTORED_MESSAGE);
    }
}
